<?php

namespace OpenEMR\Tests\Unit\ZendModules\FHIR\Listener;

use OpenEMR\Common\Uuid\UuidMapping;
use OpenEMR\Common\Uuid\UuidRegistry;
use OpenEMR\Events\Services\ServiceSaveEvent;
use OpenEMR\Services\FHIR\Observation\FhirObservationSocialHistoryService;
use OpenEMR\Services\SocialHistoryService;
use OpenEMR\Services\VitalsService;
use OpenEMR\ZendModules\FHIR\Listener\UuidMappingEventsSubscriber;
use PHPUnit\Framework\TestCase;

class UuidMappingEventsSubscriberTest extends TestCase
{
    private $uuidRecordsToDelete = [];

    protected function setUp(): void
    {
        parent::setUp(); // TODO: Change the autogenerated stub
        $this->uuidRecordsToDelete = [];
    }

    protected function tearDown(): void
    {
        parent::tearDown(); // TODO: Change the autogenerated stub
        if (!empty($this->uuidRecordsToDelete)) {
            foreach ($this->uuidRecordsToDelete as $record) {
                $sql = "DELETE FROM uuid_registry WHERE uuid IN (select uuid FROM uuid_mapping WHERE target_uuid = ? )";
                $sqlMapping = "DELETE FROM uuid_mapping WHERE target_uuid = ? ";
                sqlStatementThrowException($sql, [$record]);
                sqlStatementThrowException($sqlMapping, [$record]);

                $sqlRegistry = "DELETE FROM uuid_registry WHERE uuid =? ";
                sqlStatementThrowException($sqlRegistry, [$record]);
            }
        }
    }

    public function testOnServicePostSaveEventForVitals()
    {
        $uuidMapping = new UuidMapping();
        $uuidRegistry = UuidRegistry::getRegistryForTable(VitalsService::TABLE_VITALS);
        $targetUuid = $uuidRegistry->createUuid();
        $this->uuidRecordsToDelete[] = $targetUuid;

        $serviceSaveEvent = new ServiceSaveEvent(new VitalsService(), ['uuid' => $targetUuid]);

        $uuidMappingSubscriber = new UuidMappingEventsSubscriber();
        $uuidMappingSubscriber->onServicePostSaveEvent($serviceSaveEvent, ServiceSaveEvent::EVENT_POST_SAVE);

        $mappedRecords = $uuidMapping->getMappedRecordsForTableUUID($targetUuid);
        $resourcePath = [];
        foreach ($mappedRecords as $record) {
            $this->assertEquals("Observation", $record['resource'], "Resource was not populated for mapped uuids");
            $resourcePath[] = $record['resource_path'];
        }
        $expectedCodes = ['59408-5', '77606-2', '59576-9', '8289-1'];
        $expectedPaths = array_map(function ($code) {
            return "category=vital-signs&code=$code";
        }, $expectedCodes);
        foreach ($expectedPaths as $path) {
            $this->assertContains($path, $resourcePath, "Mapped UUIDs did not contain $path");
        }
    }

    public function testOnServicePostSaveEventForSocialHistory()
    {
        $uuidMapping = new UuidMapping();
        $uuidRegistry = UuidRegistry::getRegistryForTable(SocialHistoryService::TABLE_NAME);
        $targetUuid = $uuidRegistry->createUuid();
        $this->uuidRecordsToDelete[] = $targetUuid;

        $serviceSaveEvent = new ServiceSaveEvent(new SocialHistoryService(), ['uuid' => $targetUuid]);

        $uuidMappingSubscriber = new UuidMappingEventsSubscriber();
        $uuidMappingSubscriber->onServicePostSaveEvent($serviceSaveEvent, ServiceSaveEvent::EVENT_POST_SAVE);

        $mappedRecords = $uuidMapping->getMappedRecordsForTableUUID($targetUuid);
        $resourcePath = [];
        foreach ($mappedRecords as $record) {
            $this->assertEquals("Observation", $record['resource'], "Resource was not populated for mapped uuids");
            $resourcePath[] = $record['resource_path'];
        }
        $expectedCodes = [FhirObservationSocialHistoryService::SMOKING_CESSATION_CODE];
        $expectedPaths = array_map(function ($code) {
            return "category=" . FhirObservationSocialHistoryService::CATEGORY . "&code=$code";
        }, $expectedCodes);
        foreach ($expectedPaths as $path) {
            $this->assertContains($path, $resourcePath, "Mapped UUIDs did not contain $path");
        }
    }
}
