<?php

namespace OpenEMR\RestControllers;

use OpenEMR\Common\Http\HttpRestController;
use OpenEMR\Common\Database\QueryUtils;

class DietOrdersRestController extends HttpRestController
{
    /**
     * GET /diet-orders
     * Fetch all diet orders, optional filter by pid
     */
    public function getAll($params)
    {
        $sql = "SELECT * FROM form_diet_orders";
        $bindings = [];
        if (!empty($params['pid'])) {
            $sql .= " WHERE pid = ?";
            $bindings[] = $params['pid'];
        }
        $sql .= " ORDER BY id DESC";
        return QueryUtils::fetchAll($sql, $bindings);
    }

    /**
     * GET /diet-orders/{id}
     * Fetch one diet order by ID
     */
    public function getOne($id, $params)
    {
        $sql = "SELECT * FROM form_diet_orders WHERE id = ?";
        return QueryUtils::fetchSingle($sql, [$id]);
    }

    /**
     * POST /diet-orders
     * Create a new diet order
     */
    public function create($params)
    {
        $required = ['pid', 'encounter', 'diet_type1'];
        foreach ($required as $r) {
            if (empty($params[$r])) {
                throw new \Exception("Missing required field: $r");
            }
        }

        $sql = "INSERT INTO form_diet_orders 
                (pid, encounter, diet_type1, diet_type2, texture_mod, supplements, allergies, additional_instructions, created_at)
                VALUES (?,?,?,?,?,?,?,?,NOW())";

        $bindings = [
            $params['pid'],
            $params['encounter'],
            $params['diet_type1'],
            $params['diet_type2'] ?? '',
            $params['texture_mod'] ?? '',
            $params['supplements'] ?? '',
            $params['allergies'] ?? '',
            $params['additional_instructions'] ?? ''
        ];

        QueryUtils::sqlInsert($sql, $bindings);

        return ["message" => "Diet order created"];
    }

    /**
     * PUT /diet-orders/{id}
     * Update an existing diet order
     */
    public function update($id, $params)
    {
        $sql = "UPDATE form_diet_orders SET 
                    diet_type1 = ?, diet_type2 = ?, texture_mod = ?, supplements = ?, allergies = ?, additional_instructions = ? 
                WHERE id = ?";

        $bindings = [
            $params['diet_type1'] ?? '',
            $params['diet_type2'] ?? '',
            $params['texture_mod'] ?? '',
            $params['supplements'] ?? '',
            $params['allergies'] ?? '',
            $params['additional_instructions'] ?? '',
            $id
        ];

        QueryUtils::sqlUpdate($sql, $bindings);
        return ["message" => "Diet order updated"];
    }

    /**
     * DELETE /diet-orders/{id}
     * Delete a diet order
     */
    public function delete($id)
    {
        $sql = "DELETE FROM form_diet_orders WHERE id = ?";
        QueryUtils::sqlDelete($sql, [$id]);
        return ["message" => "Diet order deleted"];
    }
}
