<?php

/**
 * CcdaGenerator.php  is responsible for generating a CCDA document.  It creates the initial xml input data for the
 * ccda node service and then communicates with the ccda node service to get back the generated ccda document.
 *
 * @package openemr
 * @link      http://www.open-emr.org
 * @author    Stephen Nielson <snielson@discoverandchange.com>
 * @copyright Copyright (c) 2022 Discover and Change <snielson@discoverandchange.com>
 * @license   https://github.com/openemr/openemr/blob/master/LICENSE GNU General Public License 3
 */

namespace Carecoordination\Model;

use Carecoordination\Controller\EncountermanagerController;
use DOMDocument;
use OpenEMR\Common\Logging\SystemLogger;
use XSLTProcessor;

class CcdaGenerator
{
    /**
     * @var EncounterccdadispatchTable
     */
    private $dispatchTable;

    /**
     * @var int
     */
    private $createdtime;

    /**
     * @var string
     */
    private $data;

    public function __construct(EncounterccdadispatchTable $table)
    {
        $this->dispatchTable = $table;
    }

    public function getEncounterccdadispatchTable(): EncounterccdadispatchTable
    {
        return $this->dispatchTable;
    }

    /**
     * Generates the CCDA document
     * @param $patient_id
     * @param $encounter_id
     * @param $sent_by
     * @param $send
     * @param $view
     * @param $emr_transfer
     * @param $components
     * @param $sections
     * @param $recipients
     * @param $params
     * @param $document_type
     * @param $referral_reason
     * @param $date_options has the format of ['date_start' => 'YYYY-MM-DD HH:mm:ss', 'date_end' => 'YYYY-MM-DD HH:mm:ss', 'filter_content' => boolean]
     * @return GeneratedCcdaResult
     * @throws \Exception
     */
    public function generate(
        $patient_id,
        $encounter_id,
        $sent_by,
        $send,
        $view,
        $emr_transfer,
        $components,
        $sections,
        $recipients,
        $params,
        $document_type,
        $referral_reason,
        $date_options = []
    ): GeneratedCcdaResult {

        // we need to make sure we don't accidently stuff in the debug logs any PHI so we'll only report on the presence of certain variables
        (new SystemLogger())->debug("CcdaGenerator->generate() called ", ['patient_id' => $patient_id
                , 'encounter_id' => $encounter_id, 'sent_by' => (!empty($sent_by) ? "sent_by not empty" : "sent_by is empty")
                , 'send' => $send, 'view' => $view, 'emr_transfer' => $emr_transfer, 'components' => $components
                , 'sections' => $sections, 'recipients' => !empty($recipients) ? "Recipients count " . (is_array($recipients) ? count($recipients) : "1") : "No recipients"
                , 'params' => $params, 'document_type' => $document_type
                , 'referral_reason' => (empty($referral_reason) ? "No referral reason" : "Has referral reason")
                , 'date_options' => $date_options]);
        if ($sent_by != '') {
            $_SESSION['authUserID'] = $sent_by;
        }

        if (!$sections) {
            $components0 = $this->getEncounterccdadispatchTable()->getCCDAComponents(0);
            foreach ($components0 as $key => $value) {
                if ($str ?? '') {
                    $str .= '|';
                } else {
                    $str = $key;
                    continue;
                }
                $str .= $key;
            }
            $sections = $str;
        }

        if (!$components) {
            $components1 = $this->getEncounterccdadispatchTable()->getCCDAComponents(1);
            foreach ($components1 as $key => $value) {
                if ($str1 ?? '') {
                    $str1 .= '|';
                } else {
                    $str1 = $key;
                    continue;
                }
                $str1 .= $key;
            }
            $components = $str1;
        }
        $data = $this->create_data(
            $patient_id,
            $encounter_id,
            $sections,
            $components,
            $recipients,
            $params,
            $document_type,
            $referral_reason,
            $send,
            $date_options
        );
        $content = $this->socket_get($data);
        $content = trim($content);
        // split content if unstructured is included from service.
        // service will send back a CDA and an auto created unstructured document
        // if CCM sends the documents(patient_files object) with data array.
        $unstructured = "";
        if (substr_count($content, '</ClinicalDocument>') == 2) {
            $d = explode('</ClinicalDocument>', $content);
            $content = $d[0] . '</ClinicalDocument>';
            $unstructured = $d[1] . '</ClinicalDocument>';
        }
        // save if generated by service. Filo!
        if (!empty($unstructured)) {
            $generatedResultUnstructered = $this->getEncounterccdadispatchTable()->logCCDA(
                $patient_id,
                $encounter_id,
                base64_encode($unstructured),
                $this->createdtime,
                0,
                $_SESSION['authUserID'],
                'unstructured',
                $view,
                $send,
                $emr_transfer
            );
        }
        $generatedResult = $this->getEncounterccdadispatchTable()->logCCDA(
            $patient_id,
            $encounter_id,
            base64_encode($content),
            $this->createdtime,
            0,
            $_SESSION['authUserID'],
            $document_type,
            $view,
            $send,
            $emr_transfer
        );

        return $generatedResult;
    }

    public function socket_get($data)
    {
        $serviceRequestor = new CcdaServiceDocumentRequestor();
        $content = $serviceRequestor->socket_get($data);
        return $content;
    }

    public function create_data($pid, $encounter, $sections, $components, $recipients, $params, $document_type, $referral_reason = null, $send = null, $date_options = [])
    {
        $modelGenerator = new CcdaServiceRequestModelGenerator($this->getEncounterccdadispatchTable());
        $modelGenerator->create_data(
            $pid,
            $encounter,
            $sections,
            $components,
            $recipients,
            $params,
            $document_type,
            $referral_reason,
            $send,
            $date_options
        );
        $this->createdtime = $modelGenerator->getCreatedTime();
        $this->data = $modelGenerator->getData();
        return $this->data;
    }
}
